using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;

namespace ProgressDialog
{
    /// <summary>
    /// This defines an interface which can be implemented by UI elements
    /// which indicate the progress of a long operation.
    /// (See ProgressWindow for a typical implementation)
    /// </summary>
    internal interface IProgressCallback
    {
        /// <summary>
        /// Call this method from the worker thread to initialize
        /// the progress callback, without setting the range
        /// </summary>
        void Begin();
        /// <summary>
        /// Call this method from the worker thread to initialize
        /// the progress callback.
        /// </summary>
        /// <param name="minimum">The minimum value in the progress range (e.g. 0)</param>
        /// <param name="maximum">The maximum value in the progress range (e.g. 100)</param>
        void Begin(int minimum, int maximum);
        /// <summary>
        /// Call this method from the worker thread to initialize
        /// the progress callback, and control whether or not the 
        /// user can cancel the operation.
        /// </summary>
        void Begin(bool AllowCancel);

        /// <summary>
        /// Call this method from the worker thread to finalize the progress meter
        /// </summary>
        /// <remarks>You must have called one of the Begin() methods prior to this call.</remarks>
        void End();

        /// <summary>
        /// Call this method from the worker thread to update the form caption.
        /// </summary>
        /// <param name="text">The caption text to display</param>
        /// <remarks>You must have called one of the Begin() methods prior to this call.</remarks>
        void SetCaption(String text);
        /// <summary>
        /// Call this method from the worker thread to update the progress text.
        /// </summary>
        /// <param name="text">The progress text to display</param>
        /// <remarks>You must have called one of the Begin() methods prior to this call.</remarks>
        void SetText(String text);

        /// <summary>
        /// Call this method from the worker thread to reset the range in the progress callback
        /// for the primary progress bar.
        /// </summary>
        /// <param name="minimum">The minimum value in the progress range (e.g. 0)</param>
        /// <param name="maximum">The maximum value in the progress range (e.g. 100)</param>
        /// <remarks>You must have called one of the Begin() methods prior to this call.</remarks>
        void SetRange_Primary(int minimum, int maximum);
        /// <summary>
        /// Call this method from the worker thread to increase the progress counter by a specified value
        /// for the primary progress bar.
        /// </summary>
        /// <param name="val">The amount by which to increment the progress indicator</param>
        /// <remarks>You must have called one of the Begin() methods prior to this call.</remarks>
        void StepTo_Primary(int val);
        /// <summary>
        /// Call this method from the worker thread to step the progress meter to a particular value
        /// for the primary progress bar.
        /// </summary>
        /// <param name="val">The value to which to step the meter</param>
        /// <remarks>You must have called one of the Begin() methods prior to this call.</remarks>
        void Increment_Primary(int val);

        /// <summary>
        /// Call this method from the worker thread to reset the range in the progress callback
        /// for the secondary progress bar.
        /// </summary>
        /// <param name="minimum">The minimum value in the progress range (e.g. 0)</param>
        /// <param name="maximum">The maximum value in the progress range (e.g. 100)</param>
        /// <remarks>You must have called one of the Begin() methods prior to this call.</remarks>
        void SetRange_Secondary(int minimum, int maximum);
        /// <summary>
        /// Call this method from the worker thread to increase the progress counter by a specified value
        /// for the secondary progress bar.
        /// </summary>
        /// <param name="val">The amount by which to increment the progress indicator</param>
        /// <remarks>You must have called one of the Begin() methods prior to this call.</remarks>
        void StepTo_Secondary(int val);
        /// <summary>
        /// Call this method from the worker thread to step the progress meter to a particular value
        /// for the secondary progress bar.
        /// </summary>
        /// <param name="val">The value to which to step the meter</param>
        /// <remarks>You must have called one of the Begin() methods prior to this call.</remarks>
        void Increment_Secondary(int val);

        /// <summary>
        /// If this property is true, then you should abort work
        /// </summary>
        /// <remarks>You must have called one of the Begin() methods prior to this call.</remarks>
        bool IsAborting
        {
            get;
        }

        bool IsInitialized
        {
            get;
        }

        int Stages
        {
            get;
        }

    }
    /// <summary>
    /// Summary description for ProgressWindow.
    /// </summary>
    internal class ProgressWindow : System.Windows.Forms.Form, IProgressCallback
    {
        private System.Windows.Forms.Button cancelButton;
        private System.Windows.Forms.Label label;
        private System.Windows.Forms.ProgressBar prgPrimary;

        /// <summary>
        /// Required designer variable.
        /// </summary>
        private System.ComponentModel.Container components = null;

        public delegate void SetTextInvoker(String text);
        public delegate void IncrementInvoker(int val);
        public delegate void StepToInvoker(int val);
        public delegate void RangeInvoker(int minimum, int maximum);
        public delegate void CancelInvoker(bool AllowCancel);

        private System.Threading.ManualResetEvent initEvent = new System.Threading.ManualResetEvent(false);
        private System.Threading.ManualResetEvent abortEvent = new System.Threading.ManualResetEvent(false);
        private String titleRoot = "";
        private bool requiresClose = true;
        private bool initialized = false;

        public ProgressWindow()
        {
            // Required for Windows Form Designer support
            InitializeComponent();
        }

        #region Implementation of IProgressCallback
        /// <summary>
        /// Call this method from the worker thread to initialize
        /// the progress callback, without setting the range
        /// </summary>
        public void Begin()
        {
            initEvent.WaitOne();
            Invoke(new MethodInvoker(DoBegin));
        }
        /// <summary>
        /// Call this method from the worker thread to initialize
        /// the progress meter.
        /// </summary>
        /// <param name="minimum">The minimum value in the progress range (e.g. 0)</param>
        /// <param name="maximum">The maximum value in the progress range (e.g. 100)</param>
        public void Begin(int minimum, int maximum)
        {
            initEvent.WaitOne();
            Invoke(new RangeInvoker(DoBegin), new object[] { minimum, maximum });
        }
        /// <summary>
        /// Call this method from the worker thread to initialize
        /// the progress meter.
        /// </summary>
        /// <param name="AllowCancel">A boolean value that determines whether or not the cancel button is displayed.</param>
        public void Begin(bool AllowCancel)
        {
            initEvent.WaitOne();
            Invoke(new CancelInvoker(DoBegin), new object[] { AllowCancel });
        }

        /// <summary>
        /// Call this method from the worker thread to finalize the progress meter
        /// </summary>
        public void End()
        {
            if (requiresClose)
            {
                Invoke(new MethodInvoker(DoEnd));
            }
        }

        /// <summary>
        /// Call this method from the worker thread to update the form's caption.
        /// </summary>
        /// <param name="text">The caption text to display</param>
        public void SetCaption(String text)
        {
            Invoke(new SetTextInvoker(DoSetCaption), new object[] { text });
        }
        /// <summary>
        /// Call this method from the worker thread to update the progress text.
        /// </summary>
        /// <param name="text">The progress text to display</param>
        public void SetText(String text)
        {
            Invoke(new SetTextInvoker(DoSetText), new object[] { text });
        }

        /// <summary>
        /// Call this method from the worker thread to reset the range in the progress callback
        /// </summary>
        /// <param name="minimum">The minimum value in the progress range (e.g. 0)</param>
        /// <param name="maximum">The maximum value in the progress range (e.g. 100)</param>
        /// <remarks>You must have called one of the Begin() methods prior to this call.</remarks>
        public void SetRange_Primary(int minimum, int maximum)
        {
            initEvent.WaitOne();
            Invoke(new RangeInvoker(DoSetRange_Primary), new object[] { minimum, maximum });
        }
        /// <summary>
        /// Call this method from the worker thread to increase the progress counter by a specified value.
        /// </summary>
        /// <param name="val">The amount by which to increment the progress indicator</param>
        public void Increment_Primary(int val)
        {
            Invoke(new IncrementInvoker(DoIncrement), new object[] { val });
        }
        /// <summary>
        /// Call this method from the worker thread to step the progress meter to a particular value.
        /// </summary>
        /// <param name="val"></param>
        public void StepTo_Primary(int val)
        {
            Invoke(new StepToInvoker(DoStepTo), new object[] { val });
        }

        /// <summary>
        /// Call this method from the worker thread to reset the range in the progress callback
        /// for the secondary progressbar
        /// </summary>
        /// <param name="minimum">The minimum value in the progress range (e.g. 0)</param>
        /// <param name="maximum">The maximum value in the progress range (e.g. 100)</param>
        /// <remarks>You must have called one of the Begin() methods prior to this call.</remarks>
        public void SetRange_Secondary(int minimum, int maximum) { return; }
        /// <summary>
        /// Call this method from the worker thread to increase the progress counter by a specified value.
        /// </summary>
        /// <param name="val">The amount by which to increment the progress indicator</param>
        public void Increment_Secondary(int val) { return; }
        /// <summary>
        /// Call this method from the worker thread to step the progress meter to a particular value.
        /// </summary>
        /// <param name="val"></param>
        public void StepTo_Secondary(int val) { return; }

        /// <summary>
        /// If this property is true, then you should abort work
        /// </summary>
        public bool IsAborting
        {
            get { return abortEvent.WaitOne(0, false); }
        }
        public bool IsInitialized
        {
            get { return initialized; }
        }
        public int Stages { get { return 1; } }
        #endregion

        #region Implementation members invoked on the owner thread
        private void DoBegin()
        {
            titleRoot = Text;
            cancelButton.Enabled = true;
            ControlBox = true;
            initialized = true;
        }
        private void DoBegin(int minimum, int maximum)
        {
            DoBegin();
            DoSetRange_Primary(minimum, maximum);
        }
        private void DoBegin(bool AllowCancel)
        {
            titleRoot = Text;
            cancelButton.Enabled = AllowCancel;
            cancelButton.Visible = AllowCancel;
            ControlBox = AllowCancel;
            prgPrimary.Width = (AllowCancel ? 192 : 273);
            initialized = true;
        }

        private void DoEnd()
        {
            Close();
        }

        private void DoSetText(String text)
        {
            label.Text = text;
        }
        private void DoSetCaption(String text)
        {
            this.Text = text;
            titleRoot = Text;
        }

        private void DoSetRange_Primary(int minimum, int maximum)
        {
            prgPrimary.Minimum = minimum;
            prgPrimary.Maximum = maximum;
            prgPrimary.Value = minimum;
        }
        private void DoIncrement(int val)
        {
            if (prgPrimary.Value < prgPrimary.Maximum)
            {
                if (val <= (prgPrimary.Maximum - prgPrimary.Value))  //If incrementing by val won't push us past the max
                    prgPrimary.Increment(val);
                else
                    prgPrimary.Value = prgPrimary.Maximum; //Otherwise, just set it to the max
            }
            UpdateStatusText();
        }
        private void DoStepTo(int val)
        {
            if (val < prgPrimary.Maximum) { prgPrimary.Value = val; }
            UpdateStatusText();
        }

        private void DoSetRange_Secondary(int minimum, int maximum) { return; }
        private void DoIncrement_Secondary(int val) { return; }
        private void DoStepTo_Secondary(int val) { return; }
        #endregion

        #region Overrides
        /// <summary>
        /// Handles the form load, and sets an event to ensure that
        /// intialization is synchronized with the appearance of the form.
        /// </summary>
        /// <param name="e"></param>
        protected override void OnLoad(System.EventArgs e)
        {
            base.OnLoad(e);
            ControlBox = false;
            initEvent.Set();
        }

        /// <summary>
        /// Clean up any resources being used.
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose(disposing);
        }

        /// <summary>Handler for 'Close' clicking</summary>
        /// <param name="e"></param>
        protected override void OnClosing(System.ComponentModel.CancelEventArgs e)
        {
            requiresClose = false;
            AbortWork();
            base.OnClosing(e);
        }
        #endregion

        #region Implementation Utilities
        /// <summary>Utility function that formats and updates the title bar text</summary>
        private void UpdateStatusText() { Text = titleRoot + String.Format(" - {0}% complete", (prgPrimary.Value * 100) / (prgPrimary.Maximum - prgPrimary.Minimum)); }
        private void cancelButton_Click(object sender, EventArgs e) { AbortWork(); }
        /// <summary>Utility function to terminate the thread</summary>
        private void AbortWork()
        {
            titleRoot = "Aborting: " + titleRoot;
            this.cancelButton.Enabled = false;
            this.cancelButton.Text = "Cancelling";
            abortEvent.Set();
        }
        #endregion

        #region Windows Form Designer generated code
        /// <summary>
        /// Required method for Designer support - do not modify
        /// the contents of this method with the code editor.
        /// </summary>
        private void InitializeComponent()
        {
            this.prgPrimary = new System.Windows.Forms.ProgressBar();
            this.label = new System.Windows.Forms.Label();
            this.cancelButton = new System.Windows.Forms.Button();
            this.SuspendLayout();
            // 
            // prgPrimary
            // 
            this.prgPrimary.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Left | System.Windows.Forms.AnchorStyles.Right)));
            this.prgPrimary.Location = new System.Drawing.Point(8, 77);
            this.prgPrimary.Margin = new System.Windows.Forms.Padding(0);
            this.prgPrimary.Name = "prgPrimary";
            this.prgPrimary.Size = new System.Drawing.Size(196, 23);
            this.prgPrimary.TabIndex = 1;
            // 
            // label
            // 
            this.label.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left)
                        | System.Windows.Forms.AnchorStyles.Right)));
            this.label.Location = new System.Drawing.Point(8, 8);
            this.label.Name = "label";
            this.label.Size = new System.Drawing.Size(276, 64);
            this.label.TabIndex = 0;
            this.label.Text = "Starting operation...";
            // 
            // cancelButton
            // 
            this.cancelButton.Anchor = System.Windows.Forms.AnchorStyles.Right;
            //this.cancelButton.DialogResult = System.Windows.Forms.DialogResult.Cancel;
            this.cancelButton.Enabled = false;
            this.cancelButton.Location = new System.Drawing.Point(210, 77);
            this.cancelButton.Margin = new System.Windows.Forms.Padding(0);
            this.cancelButton.Name = "cancelButton";
            this.cancelButton.Size = new System.Drawing.Size(75, 23);
            this.cancelButton.TabIndex = 3;
            this.cancelButton.Text = "Cancel";
            this.cancelButton.Click += new EventHandler(cancelButton_Click);
            // 
            // ProgressWindow
            // 
            this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
            this.ClientSize = new System.Drawing.Size(294, 108);
            this.Controls.Add(this.cancelButton);
            this.Controls.Add(this.prgPrimary);
            this.Controls.Add(this.label);
            this.FormBorderStyle = System.Windows.Forms.FormBorderStyle.FixedDialog;
            this.MaximizeBox = false;
            this.MinimizeBox = false;
            this.Name = "ProgressWindow";
            this.ShowInTaskbar = false;
            this.StartPosition = System.Windows.Forms.FormStartPosition.CenterParent;
            this.Text = "Single Stage ProgressWindow";
            this.ResumeLayout(false);

        }
        #endregion


    }
    internal class TwoStageProgressWindow : System.Windows.Forms.Form, IProgressCallback
    {
        private System.Windows.Forms.Button cancelButton;
        private System.Windows.Forms.Label label;
        private System.Windows.Forms.ProgressBar prgPrimary;
        private System.Windows.Forms.ProgressBar prgSecondary;

        /// <summary>
        /// Required designer variable.
        /// </summary>
        private System.ComponentModel.Container components = null;

        public delegate void SetTextInvoker(String text);
        public delegate void IncrementInvoker(int val);
        public delegate void StepToInvoker(int val);
        public delegate void RangeInvoker(int minimum, int maximum);
        public delegate void CancelInvoker(bool AllowCancel);

        private System.Threading.ManualResetEvent initEvent = new System.Threading.ManualResetEvent(false);
        private System.Threading.ManualResetEvent abortEvent = new System.Threading.ManualResetEvent(false);
        private String titleRoot = "";
        private bool requiresClose = true;
        private bool initialized = false;

        public TwoStageProgressWindow()
        {
            // Required for Windows Form Designer support
            InitializeComponent();
        }

        #region Implementation of IProgressCallback
        /// <summary>
        /// Call this method from the worker thread to initialize
        /// the progress callback, without setting the range
        /// </summary>
        public void Begin()
        {
            initEvent.WaitOne();
            Invoke(new MethodInvoker(DoBegin));
        }
        /// <summary>
        /// Call this method from the worker thread to initialize
        /// the progress meter.
        /// </summary>
        /// <param name="minimum">The minimum value in the progress range (e.g. 0)</param>
        /// <param name="maximum">The maximum value in the progress range (e.g. 100)</param>
        public void Begin(int minimum, int maximum)
        {
            initEvent.WaitOne();
            Invoke(new RangeInvoker(DoBegin), new object[] { minimum, maximum });
        }
        /// <summary>
        /// Call this method from the worker thread to initialize
        /// the progress meter.
        /// </summary>
        /// <param name="AllowCancel">A boolean value that determines whether or not the cancel button is displayed.</param>
        public void Begin(bool AllowCancel)
        {
            initEvent.WaitOne();
            Invoke(new CancelInvoker(DoBegin), new object[] { AllowCancel });
        }

        /// <summary>
        /// Call this method from the worker thread to finalize the progress meter
        /// </summary>
        public void End()
        {
            if (requiresClose)
            {
                Invoke(new MethodInvoker(DoEnd));
            }
        }

        /// <summary>
        /// Call this method from the worker thread to update the form's caption.
        /// </summary>
        /// <param name="text">The caption text to display</param>
        public void SetCaption(String text)
        {
            Invoke(new SetTextInvoker(DoSetCaption), new object[] { text });
        }
        /// <summary>
        /// Call this method from the worker thread to update the progress text.
        /// </summary>
        /// <param name="text">The progress text to display</param>
        public void SetText(String text)
        {
            Invoke(new SetTextInvoker(DoSetText), new object[] { text });
        }

        /// <summary>
        /// Call this method from the worker thread to reset the range in the progress callback
        /// </summary>
        /// <param name="minimum">The minimum value in the progress range (e.g. 0)</param>
        /// <param name="maximum">The maximum value in the progress range (e.g. 100)</param>
        /// <remarks>You must have called one of the Begin() methods prior to this call.</remarks>
        public void SetRange_Primary(int minimum, int maximum)
        {
            initEvent.WaitOne();
            Invoke(new RangeInvoker(DoSetRange_Primary), new object[] { minimum, maximum });
        }
        /// <summary>
        /// Call this method from the worker thread to increase the progress counter by a specified value.
        /// </summary>
        /// <param name="val">The amount by which to increment the progress indicator</param>
        public void Increment_Primary(int val)
        {
            Invoke(new IncrementInvoker(DoIncrement), new object[] { val });
        }
        /// <summary>
        /// Call this method from the worker thread to step the progress meter to a particular value.
        /// </summary>
        /// <param name="val"></param>
        public void StepTo_Primary(int val)
        {
            Invoke(new StepToInvoker(DoStepTo), new object[] { val });
        }

        /// <summary>
        /// Call this method from the worker thread to reset the range in the progress callback
        /// for the secondary progressbar
        /// </summary>
        /// <param name="minimum">The minimum value in the progress range (e.g. 0)</param>
        /// <param name="maximum">The maximum value in the progress range (e.g. 100)</param>
        /// <remarks>You must have called one of the Begin() methods prior to this call.</remarks>
        public void SetRange_Secondary(int minimum, int maximum)
        {
            initEvent.WaitOne();
            Invoke(new RangeInvoker(DoSetRange_Secondary), new object[] { minimum, maximum });
        }
        /// <summary>
        /// Call this method from the worker thread to increase the progress counter by a specified value.
        /// </summary>
        /// <param name="val">The amount by which to increment the progress indicator</param>
        public void Increment_Secondary(int val)
        {
            Invoke(new IncrementInvoker(DoIncrement_Secondary), new object[] { val });
        }
        /// <summary>
        /// Call this method from the worker thread to step the progress meter to a particular value.
        /// </summary>
        /// <param name="val"></param>
        public void StepTo_Secondary(int val)
        {
            Invoke(new StepToInvoker(DoStepTo_Secondary), new object[] { val });
        }

        /// <summary>
        /// If this property is true, then you should abort work
        /// </summary>
        public bool IsAborting
        {
            get { return abortEvent.WaitOne(0, false); }
        }
        public bool IsInitialized
        {
            get { return initialized; }
        }
        public int Stages { get { return 2; } }
        #endregion

        #region Implementation members invoked on the owner thread
        private void DoBegin()
        {
            titleRoot = Text;
            cancelButton.Enabled = true;
            ControlBox = true;
            initialized = true;
        }
        private void DoBegin(int minimum, int maximum)
        {
            DoBegin();
            DoSetRange_Primary(minimum, maximum);
        }
        private void DoBegin(bool AllowCancel)
        {
            titleRoot = Text;
            cancelButton.Enabled = AllowCancel;
            cancelButton.Visible = AllowCancel;
            ControlBox = AllowCancel;
            prgPrimary.Width = (AllowCancel ? 192 : 273);
            prgSecondary.Width = prgPrimary.Width;
            initialized = true;
        }

        private void DoEnd()
        {
            Close();
        }

        private void DoSetText(String text)
        {
            label.Text = text;
        }
        private void DoSetCaption(String text)
        {
            this.Text = text;
            titleRoot = Text;
        }

        private void DoSetRange_Primary(int minimum, int maximum)
        {
            prgPrimary.Minimum = minimum;
            prgPrimary.Maximum = maximum;
            prgPrimary.Value = minimum;
        }
        private void DoIncrement(int val)
        {
            if (prgPrimary.Value < prgPrimary.Maximum)
            {
                if (val <= (prgPrimary.Maximum - prgPrimary.Value))  //If incrementing by val won't push us past the max
                    prgPrimary.Increment(val);
                else
                    prgPrimary.Value = prgPrimary.Maximum; //Otherwise, just set it to the max
            }
            UpdateStatusText();
        }
        private void DoStepTo(int val)
        {
            if (val < prgPrimary.Maximum) { prgPrimary.Value = val; }
            UpdateStatusText();
        }

        private void DoSetRange_Secondary(int minimum, int maximum)
        {
            prgSecondary.Minimum = minimum;
            prgSecondary.Maximum = maximum;
            prgSecondary.Value = minimum;
        }
        private void DoIncrement_Secondary(int val)
        {
            if (prgSecondary.Value < prgSecondary.Maximum)
            {
                if (val <= (prgSecondary.Maximum - prgSecondary.Value))  //If incrementing by val won't push us past the max
                    prgSecondary.Increment(val);
                else
                    prgSecondary.Value = prgSecondary.Maximum; //Otherwise, just set it to the max
            }
        }
        private void DoStepTo_Secondary(int val) { if (val < prgSecondary.Maximum) { prgSecondary.Value = val; } }
        #endregion

        #region Overrides
        /// <summary>
        /// Handles the form load, and sets an event to ensure that
        /// intialization is synchronized with the appearance of the form.
        /// </summary>
        /// <param name="e"></param>
        protected override void OnLoad(System.EventArgs e)
        {
            base.OnLoad(e);
            ControlBox = false;
            initEvent.Set();
        }

        /// <summary>
        /// Clean up any resources being used.
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose(disposing);
        }

        /// <summary>Handler for 'Close' clicking</summary>
        /// <param name="e"></param>
        protected override void OnClosing(System.ComponentModel.CancelEventArgs e)
        {
            requiresClose = false;
            AbortWork();
            base.OnClosing(e);
        }
        #endregion

        #region Implementation Utilities
        /// <summary>Utility function that formats and updates the title bar text</summary>
        private void UpdateStatusText() { Text = titleRoot + String.Format(" - {0}% complete", (prgPrimary.Value * 100) / (prgPrimary.Maximum - prgPrimary.Minimum)); }
        private void cancelButton_Click(object sender, EventArgs e) { AbortWork(); }

        /// <summary>Utility function to terminate the thread</summary>
        private void AbortWork()
        {
            titleRoot = "Aborting: " + titleRoot;
            this.cancelButton.Enabled = false;
            this.cancelButton.Text = "Cancelling";
            abortEvent.Set();
        }
        #endregion

        #region Windows Form Designer generated code
        /// <summary>
        /// Required method for Designer support - do not modify
        /// the contents of this method with the code editor.
        /// </summary>
        private void InitializeComponent()
        {
            this.prgPrimary = new System.Windows.Forms.ProgressBar();
            this.label = new System.Windows.Forms.Label();
            this.cancelButton = new System.Windows.Forms.Button();
            this.prgSecondary = new System.Windows.Forms.ProgressBar();
            this.SuspendLayout();
            // 
            // prgPrimary
            // 
            this.prgPrimary.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Left | System.Windows.Forms.AnchorStyles.Right)));
            this.prgPrimary.Location = new System.Drawing.Point(11, 76);
            this.prgPrimary.Name = "prgPrimary";
            this.prgPrimary.Size = new System.Drawing.Size(196, 23);
            this.prgPrimary.TabIndex = 1;
            // 
            // label
            // 
            this.label.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left)
                        | System.Windows.Forms.AnchorStyles.Right)));
            this.label.Location = new System.Drawing.Point(8, 8);
            this.label.Name = "label";
            this.label.Size = new System.Drawing.Size(276, 64);
            this.label.TabIndex = 0;
            this.label.Text = "Starting operation...";
            // 
            // cancelButton
            // 
            this.cancelButton.Anchor = System.Windows.Forms.AnchorStyles.Right;
            //            this.cancelButton.DialogResult = System.Windows.Forms.DialogResult.Cancel;
            this.cancelButton.Enabled = false;
            this.cancelButton.Location = new System.Drawing.Point(213, 89);
            this.cancelButton.Name = "cancelButton";
            this.cancelButton.Size = new System.Drawing.Size(75, 23);
            this.cancelButton.TabIndex = 3;
            this.cancelButton.Text = "Cancel";
            this.cancelButton.Click += new System.EventHandler(this.cancelButton_Click);
            // 
            // prgSecondary
            // 
            this.prgSecondary.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Left | System.Windows.Forms.AnchorStyles.Right)));
            this.prgSecondary.Location = new System.Drawing.Point(11, 105);
            this.prgSecondary.Name = "prgSecondary";
            this.prgSecondary.Size = new System.Drawing.Size(196, 23);
            this.prgSecondary.TabIndex = 2;
            // 
            // TwoStageProgressWindow
            // 
            this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
            this.ClientSize = new System.Drawing.Size(294, 138);
            this.Controls.Add(this.prgSecondary);
            this.Controls.Add(this.cancelButton);
            this.Controls.Add(this.prgPrimary);
            this.Controls.Add(this.label);
            this.FormBorderStyle = System.Windows.Forms.FormBorderStyle.FixedDialog;
            this.MaximizeBox = false;
            this.MinimizeBox = false;
            this.Name = "TwoStageProgressWindow";
            this.ShowInTaskbar = false;
            this.StartPosition = System.Windows.Forms.FormStartPosition.CenterParent;
            this.Text = "Two Stage ProgressWindow";
            this.ResumeLayout(false);

        }
        #endregion

    }
}
